package com.primeton.eos.microapp.core.util;

import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 生成 UUID v7 的工具类（基于时间戳 + 随机数）
 */
public final class UUIDv7 {
    private static final int VERSION = 0x7; // v7

    private UUIDv7() {
    }

    /**
     * 生成 UUID v7
     *
     * @return UUID 对象
     */
    public static UUID randomUUID() {
        byte[] bytes = new byte[16];
        ThreadLocalRandom rnd = ThreadLocalRandom.current();

        // 1) timestamp 48 bits (millis)
        long unixMillis = System.currentTimeMillis();
        // 保持低 48 bit
        long ts48 = unixMillis & 0x0000FFFFFFFFFFFFL;

        // 写入前 6 个字节（big-endian）
        for (int i = 0; i < 6; i++) {
            bytes[i] = (byte) ((ts48 >>> (8 * (5 - i))) & 0xFF);
        }

        // 2) 16 bits: version (high 4 bits) + 12 bits random
        int rand12 = rnd.nextInt(1 << 12); // 0..4095
        int versionAndRand = ((VERSION & 0x0F) << 12) | (rand12 & 0x0FFF);
        bytes[6] = (byte) ((versionAndRand >>> 8) & 0xFF);
        bytes[7] = (byte) (versionAndRand & 0xFF);

        // 3) 16 bits: clock sequence (set variant bits to RFC4122: "10" in top two bits)
        int rand14 = rnd.nextInt(1 << 14); // 0..16383
        int clockSeq = (rand14 & 0x3FFF) | 0x8000; // set bits 15..14 as 10
        bytes[8] = (byte) ((clockSeq >>> 8) & 0xFF);
        bytes[9] = (byte) (clockSeq & 0xFF);

        // 4) 48 bits node random
        for (int i = 10; i < 16; i++) {
            bytes[i] = (byte) rnd.nextInt(0, 256);
        }

        // 构造 UUID(msb, lsb)
        ByteBuffer bb = ByteBuffer.wrap(bytes);
        long msb = bb.getLong(0);
        long lsb = bb.getLong(8);
        return new UUID(msb, lsb);
    }

    /**
     * 从 v7 UUID 中提取出时间戳（毫秒级）
     *
     * @param uuid UUID 对象
     * @return 间戳毫秒（或 null 如果无法识别）
     */
    public static Long extractTimestamp(UUID uuid) {
        Objects.requireNonNull(uuid);
        // 把 msb 的前 48 bits 提取出来
        long msb = uuid.getMostSignificantBits();
        // msb 是 signed long，但我们只关心最高 64 bits，取前 48 bits：
        // 去掉低 16 位（version+rand）
        return (msb >>> 16) & 0x0000FFFFFFFFFFFFL;
    }
}
